#!/usr/bin/env bash
set -euo pipefail

WORKDIR="$HOME/ecz-watchhouse-workdir"
ECZ_USER="ecz-gebruiker"
WIZ_DIR="/home/${ECZ_USER}/birdnet-wifi-setup"
NM_CONN_SRC="$WORKDIR/wifi-config/ECZ-BirdNET-PI-SETUP.nmconnection"
NM_CONN_DST="/etc/NetworkManager/system-connections/ECZ-BirdNET-PI-SETUP.nmconnection"
CADDYFILE="/etc/caddy/Caddyfile"
SERVICE_SRC="$WORKDIR/systemd/ecz-setup-wizard.service"
SERVICE_DST="/etc/systemd/system/ecz-setup-wizard.service"

log(){ echo -e "\n[+] $*"; }

# 0) root
if [[ $EUID -ne 0 ]]; then echo "Run met sudo"; exit 1; fi

# 1) deps
log "Packages…"
apt-get update
DEBIAN_FRONTEND=noninteractive apt-get install -y network-manager caddy python3 python3-pip

# 2) user
if ! id -u "$ECZ_USER" &>/dev/null; then
  log "Maak user ${ECZ_USER}…"
  adduser --disabled-password --gecos "" "$ECZ_USER"
  usermod -aG sudo,audio,video,plugdev,netdev "$ECZ_USER"
fi

# 3) wizard
log "Deploy wizard → ${WIZ_DIR}"
mkdir -p "$WIZ_DIR"
rsync -a "$WORKDIR/wizard/" "$WIZ_DIR/"
chown -R "$ECZ_USER:$ECZ_USER" "$WIZ_DIR"
# safety: pad ‘shaady’ fixen als het er nog in staat
sed -i 's#/home/shaady#/home/ecz-gebruiker#g' "$WIZ_DIR/app.py" || true

# 4) python deps
log "Python deps (Flask)…"
pip3 install --break-system-packages Flask==3.0.3 Jinja2 itsdangerous Werkzeug || pip3 install Flask==3.0.3

# 5) systemd service
log "Install service…"
install -m 0644 "$SERVICE_SRC" "$SERVICE_DST"
systemctl daemon-reload
systemctl enable --now ecz-setup-wizard

# 6) NetworkManager open AP
log "Hotspot profiel plaatsen…"
install -m 0600 -o root -g root "$NM_CONN_SRC" "$NM_CONN_DST"
nmcli connection load "$NM_CONN_DST" >/dev/null || true
systemctl restart NetworkManager
nmcli dev set wlan0 managed yes 2>/dev/null || true
ip link set wlan0 up 2>/dev/null || true
nmcli con up "ECZ-BirdNET-PI-SETUP" ifname wlan0 || true

# 7) Caddy captive redirect (voeg snippet toe als nog niet aanwezig)
log "Caddy captive redirect configureren…"
if [[ ! -f "$CADDYFILE" ]]; then
  echo 'http:// { respond "Caddy draait" 200 }' > "$CADDYFILE"
fi
if ! grep -q "generate_204" "$CADDYFILE"; then
  cat "$WORKDIR/docs/Caddy-captive-snippet.txt" >> "$CADDYFILE"
fi
caddy validate --config "$CADDYFILE"
systemctl reload caddy || systemctl restart caddy

# 8) checks
log "Status:"
nmcli -t -f DEVICE,TYPE,STATE,CONNECTION device status || true
echo "wlan0 IP: $(ip -4 addr show wlan0 | awk '/inet /{print $2}' | cut -d/ -f1 || true)"
systemctl --no-pager --full status ecz-setup-wizard | sed -n '1,12p' || true

echo
echo "✅ Klaar. Verbind met SSID: ECZ-BirdNET-PI-SETUP (open)"
echo "➡️ Captive/handmatig: http://10.42.0.1:1234"
